/**
 * ImageViewer Class
 * Version: 1.0.3
 * Author: Colin Moock
 * Updates: http://www.moock.org/asdg/codedepot/
 * ActionScript 2.0 Version: http://www.moock.org/as2e/examples/
 *
 * An on-screen rectangular region for displaying, zooming, and panning an image.
 * Supports preloading for externally loaded images. To pan an image, click and
 * drag it with the mouse. To scale an image up or down, use the up or down arrow
 * keys (pressing up or down scales the image in *all* viewers currently on 
 * screen). This class relies on the custom MovieClip.drawRect() method.
 *
 * Constructor params:
 *   target           The movieclip in which to create the viewer.
 *   depth            The depth in target on which to place the viewer.
 *   x                The horizontal position of the viewer.
 *   y                The vertical position of the viewer.
 *   w                The width of the viewer, in pixels.
 *   h                The height of the viewer, in pixels.
 *   borderThickness  The pixel thickness of the border around the viewer.
 *   borderColor      The numeric RGB color of the border around the viewer.
 *   allowZoom        Boolean. Indicates whether the image can be zoomed 
 *                    via the arrow keys. Defaults to true (zoom enabled).
 *   allowPan         Boolean. Indicates whether the image can be panned 
 *                    via a mouse drag. Defaults to true (pan enabled). 
 *
 * Methods:
 *   loadImage()      Retrieve an image to view.
 *   centerImage()    Centers the image in the image viewer.
 *   zoomImage()      Enlarges or shrinks the image.
 *   setScaleMode()   Determines how the image is scaled upon loading.
 *   getScaleMode()   Returns the current scale mode setting.
 *   enableZoom()     Turns on zoom feature.
 *   disableZoom()    Turns off zoom feature. 
 *   enablePan()      Turns on pan feature. 
 *   disablePan()     Turns on pan feature.  
 *
 * Extension ideas:
 *   1) Add a listener event for image load completion.
 *   2) Use a FScrollPane component instead of border_mc and mask_mc.
 *   3) Change the zoom controls to allow independent zooming of
 *      multiple images. One approach would be to implement input focus
 *      and base zooming on the currently focussed viewer.
 *   4) Add an option to disallow zooming out so far that the viewer
 *      background appears.
 *   5) Add option to have an opaque viewer background.
 */
 
 // Load supporting method
 #include "MovieClip.drawRect.as"
 
 // Class constructor
_global.ImageViewer = function (target, 
                                depth, 
                                x, 
                                y, 
                                w, 
                                h,
                                borderThickness,
                                borderColor,
                                allowZoom,
                                allowPan) {
  /*** PROPERTIES ***/                                
  // Holds the reference to the container_mc clip.
  this.container = null;
  
  // Stores the border thickness.
  this.borderThickness = borderThickness;
  
  // Stores the border color.
  this.borderColor = borderColor;
  
  // Stores the image/viewer scale mode.
  // By default images are not scaled when loaded.
  this.scaleMode = "noScale";  // For possible values, see setScaleMode()
  
  // Stores the clip in which the viewer is created.
  this.target_mc = target;

  // Stores whether zooming is allowed.
  // If allowZoom wasn't specified, set it to true.
  this.allowZoom = typeof allowZoom == "boolean" ? allowZoom : true;

  // Stores whehter panning is allowed.
  // If allowPan wasn't specified, set it to true.
  this.allowPan = typeof allowPan == "boolean" ? allowPan : true;
  
  // Stores whether zooming is currently enabled.
  this.zoomEnabled = false;
  
  // Stores whether panning is currently enabled.
  this.panEnabled = false;
  
  /*** START INITIALIZATION ***/  
  // Store a reference to the current ImageViewer object.
  // Later on in the constructor, we'll use this reference from the 
  // inner callback functions assigned to the image holder clip.
  var imgViewer = this;
  
  /*** MOVIE CLIP CREATION ***/    
  // Create and position a container clip to hold the viewer.
  this.target_mc.createEmptyMovieClip("container_mc" + depth, depth);
  this.container = this.target_mc["container_mc" + depth];
  this.container._x = x;
  this.container._y = y;
  
  // In the container, create a clip to hold the image clip. A new image clip 
  // is created by the loadImage() method every time a new image is loaded.
  this.container.createEmptyMovieClip("imageHolder_mc", 0);
  
  // A NOTE ABOUT THE IMAGEHOLDER_MC CLIP:  
  // You might think we'd just have a single clip that contains the image 
  // directly instead of a parent clip (imageHolder_mc) that contains a
  // clip to hold the image (imageHolder_mc.image_mc). Having a single
  // clip (image_mc) would mostly work, but it would also prevent us 
  // from setting properties on the image_mc clip because properties 
  // are wiped out when a JPEG file loads into a clip. 
  // Hence, to be able to set persistent properties on the image 
  // (that are not lost when a JPEG loads), we contain
  // the image clip (image_mc) in a parent clip (imageHolder_mc), and we 
  // set the properties on that parent. This lets us make the image invisible 
  // while it loads (by setting imageHolder_mc._visible to false.)
  // Let's do that next...

  // Hide the image holder so the image doesn't appear until 
  // we've positioned and scaled it.
  this.container.imageHolder_mc._visible = false;

  // In the container, create a clip to hold the border around the image.
  // Put the border on depth 1, above the image.
  this.container.createEmptyMovieClip("border_mc", 1);
  
  // Draw a rectangular outline in the border clip, with the
  // specified dimensions and color.
  this.container.border_mc.lineStyle(borderThickness, borderColor);
  this.container.border_mc.drawRect(0, 0, w, h);
  
  // In the container, create a clip to act as the mask over the image.
  this.container.createEmptyMovieClip("mask_mc", 2);

  // Draw a rectangle in the mask.
  this.container.mask_mc.drawRect(0, 0, w, h, 0x0000FF, 100);
  
  // Hide the mask (it will still function as a mask when invisible).
  this.container.mask_mc._visible = false; 

  /*** DONE CONSTRUCTING ***/  
}


/**
 * Method: ImageViewer.loadImage()
 *   Desc: Loads a .jpeg file into the image viewer.
 * Params:
 *   URL   The location of the .jpeg file.
 */
_global.ImageViewer.prototype.loadImage = function (URL) {
  // Remove any previous image_mc clip.
  if (typeof this.container.imageHolder_mc.image_mc == "movieclip") {
    this.container.imageHolder_mc.image_mc.removeMovieClip();
  }
  
  // In the clip that holds the image, make a child clip into which
  // we'll actually load the image. 
  this.container.imageHolder_mc.createEmptyMovieClip("image_mc", 1);

  // Store local references to the current image viewer object
  // and the image clip. These will be accessed by inner functions via
  // the scope chain.
  var imgViewer = this;
  var image = this.container.imageHolder_mc.image_mc;

  // The interval identifier for the preload interval.
  var loadCheckID;

  // Now load the image.
  image.loadMovie(URL);

  // Start preloading the image.
  loadCheckID = setInterval(preloadImage, 100, getTimer());

  // Create a loading status text field to show the user the load progress.
  // Put it on depth 3, above the image.
  this.container.createTextField("loadStatus_txt", 3, 0, 0, 0, 0);
  this.container.loadStatus_txt.background = true;
  this.container.loadStatus_txt.border = true;
  this.container.loadStatus_txt.setNewTextFormat(new TextFormat("Arial, Helvetica, _sans", 10, imgViewer.borderColor, false, false, false, null, null, "right"));
  this.container.loadStatus_txt.autoSize = "left";
  this.container.loadStatus_txt.text = "LOADING";

  // Position the load status text field
  this.container.loadStatus_txt._y = 3;
  this.container.loadStatus_txt._x = 3;

  // Next comes the preloader inner function, which is called every 100ms by
  // setInterval(). During loading, preloadImage() displays the image load 
  // progress. When loading completes, preloadImage() sizes and positions 
  // the image, turns on its mask, and removes the load status message.
  function preloadImage (startLoadTime) {
    // Stores the greater of the border's width or height. Used to scale the
    // image to the border if scaleMode is set to scaleImageToViewer.
    var largestBorderDimension;

    // Stores the amount the border is larger or smaller than the image.
    // Used to scale the image to the border if scaleMode is 
    // set to scaleImageToViewer.    
    var imageScaleFactor;

    // Display a loading message in the loadStatus_txt text field, but
    // only if the bytes total has been properly determined. (An empty
    // clip created at author-time has a bytes total of 4. An empty clip
    // created via createEmptyMovieClip() has a bytes total of 0.
    // If the image isn't found, bytes total will be -1. Hence if bytes total 
    // is greater than 4, we know the image has definitely started 
    // loading (and may even be done loading, but we'll check for that 
    // separately). Technically, we could simply check that bytes total 
    // is > 0 (because we're not using author-time clips), but in the event
    // that someone changes this class to use author-time clips, we don't want
    // the preloader to break, so we use > 4.
    if (image.getBytesTotal() > 4) {
      imgViewer.container.loadStatus_txt.text = "LOADING: " 
        + Math.floor(image.getBytesLoaded() / 1024)
        + "/" + Math.floor(image.getBytesTotal() / 1024) + " KB";
    }
      
    // The empty clip into which we'll load the image has a byte size of 0.
    // We know the image is fully loaded when there are more bytes than an
    // empty clip AND the number of bytes loaded is the same as the total
    // number of bytes. (See the preceding comment for an explanation of 
    // the need for "> 4").
    if (image.getBytesTotal() > 4 && image.getBytesLoaded() == image.getBytesTotal()) {
      // (Side note...we know the image has loaded, so now would be a nice 
      // time to notify listeners that the image finished loading. 
      // We'll leave listener events out for this version.)

      // We're done loading, so stop calling preloadImage().
      clearInterval(loadCheckID);
            
      // And remove the loading message.
      imgViewer.container.loadStatus_txt.removeTextField();
     
      // Now the fun part! Position and size the image depending on 
      // the scale mode.
      if (imgViewer.getScaleMode() == "noScale") {
        // If the imgViewer is set to "noScale" mode, simply center
        // the image in the image viewer.
        imgViewer.centerImage();
      } else if (imgViewer.getScaleMode() == "scaleImageToViewer") {
        // If the imgViewer is in "scaleImageToViewer" mode, determine
        // the largest dimension of the image viewer border. We'll match
        // the image's corresponding dimension to the border size.
        if (imgViewer.container.mask_mc._width >= 
            imgViewer.container.mask_mc._height) {
          largestBorderDimension = "_width";
        } else {
          largestBorderDimension = "_height";
        }
        
        // Calculate the amount the border is larger or smaller than the image,
        // as a percentage. Then apply that scale factor to the image.
        imageScaleFactor = imgViewer.container.mask_mc[largestBorderDimension]
                           / image[largestBorderDimension];
        image._xscale = image._yscale *= imageScaleFactor;

        // The image is scaled to fit the border. Now center the image.
        imgViewer.centerImage();
      } else if (imgViewer.getScaleMode() == "scaleViewerToImage") {
        // If the imgViewer is in "scaleViewerToImage" mode, enlarge the border
        // and the mask to the size of the image.
        imgViewer.container.border_mc.clear();
        imgViewer.container.mask_mc.clear();
        imgViewer.container.border_mc.lineStyle(imgViewer.borderThickness,
                                                imgViewer.borderColor);
        imgViewer.container.border_mc.drawRect(0, 0, 
                                               image._width, image._height);
        imgViewer.container.mask_mc.drawRect(0, 0, image._width, 
                                             image._height, 0xFFFFFF, 100);      
      }

      // Apply the mask. In order to take effect, this must be done 
      // after the image has loaded and the mask has been drawn.
      imgViewer.container.imageHolder_mc.setMask(imgViewer.container.mask_mc);
      
      // We're all done sizing and positioning the image...now show it.
      imgViewer.container.imageHolder_mc._visible = true;
      
      // Attempt to turn on panning and zooming. Each enable method
      // will worry about whether those features are 
      // allowed for this ImageViewer.
      imgViewer.enablePan();
      imgViewer.enableZoom();

    } else if ((image.getBytesTotal() < 4)
               && (getTimer() - startLoadTime > 7000)) {
      // If the clip has a bytes total of less than 4 after 7
      // seconds, we can assume that either there's no network connection
      // or the image wasn't found.
      imgViewer.container.loadStatus_txt.text = "ERROR LOADING IMAGE";
      clearInterval(loadCheckID);
    }
  }
}

/**
 * Method: ImageViewer.enablePan()
 *   Desc: Turns on panning.
 */
_global.ImageViewer.prototype.enablePan = function () {
  // If panning isn't allowed for this ImageViewer or no image
  // exists, or panning is already on, abort.
  if (!this.allowPan 
      || typeof this.container.imageHolder_mc.image_mc != "movieclip"
      || this.panEnabled == true) {
    return false;
  }
  
  // Create a convenience reference to the image_mc clip. We'll access it
  // via the scope chain from the inner callback functions defined next.
  var image = this.container.imageHolder_mc.image_mc;

  // This ImageViewer object will listen for mouse down and mouse up events.
  // Java developers should note that even though Mouse defines three events,
  // we're not obliged to implement them all (specifically, we leave out
  // onMouseMove()).  We don't want to respond to mouse movements, so we just 
  // leave onMouseMove() unimplemented.
  
  // Here's our onMouseDown() callback. When the mouse is clicked down over 
  // the image, we'll start dragging it.
  this.onMouseDown = function () {
    // A convenient reference to the mask clip.
    var mask = this.container.mask_mc;
  
    // If the mouse was clicked over the image...
    if (mask.hitTest(_level0._xmouse, _level0._ymouse, false)) {
      // Start dragging the image, but constrain the dragging region to the
      // mask area so the image can't be dragged completely out of view.
      image.startDrag(false, 
                   -image._width + mask._width,
                   -image._height + mask._height,
                   0,
                   0);
    }
  }

  // Here's our onMouseUp() callback. When the mouse is released, 
  // stop dragging the image.
  this.onMouseUp = function () {
    // stopDrag() cancels any drag operation in progress, and
    // is harmless to invoke if no drag is in progress.
    stopDrag();
  }

  // Tell the Mouse object that the image viewer (this) wants to listen
  // for Mouse events.
  Mouse.addListener(this);
  
  // Make a note that panning is enabled.
  this.panEnabled = true;
}

/**
 * Method: ImageViewer.disablePan()
 *   Desc: Turns off panning.
 */
_global.ImageViewer.prototype.disablePan = function () {
  // Make a note that panning is disabled.
  this.panEnabled = false;

  // Tell the Mouse object that the image viewer (this) no longer
  // wants to listen for Mouse events.
  return Mouse.removeListener(this);
}

/**
 * Method: ImageViewer.enableZoom()
 *   Desc: Turns on zooming.
 */
_global.ImageViewer.prototype.enableZoom = function () {
  // If zooming isn't allowed for this ImageViewer or no image
  // exists, or zooming is already on, abort.
  if (!this.allowZoom 
      || typeof this.container.imageHolder_mc.image_mc != "movieclip"
      || this.zoomEnabled == true) {
    return false;
  }

  // Create a convenience reference to the image_mc clip. We'll access it
  // via the scope chain from the inner callback functions defined next.
  var image = this.container.imageHolder_mc.image_mc;
  
  // The image viewer object will listen for key down events.
  // When the UP arrow key is pressed, scale the image up. When the 
  // DOWN arrow key is pressed, scale the image down.
  this.onKeyDown = function () {
    // If the last key pressed was...
    if (Key.getCode() == Key.UP) {  // ...the UP arrow...
      // ...scale the image up.
      this.zoomImage(1.1);
    } else if (Key.getCode() == Key.DOWN) {  // ...the DOWN arrow...
      // ...scake the image down.
      // Note that "1 - (.1/1.1)" is the inverse zoom factor. E.g., for a
      // 1.5 times zoom in, the inverse zoom out is .66 (1 - .5/1.5).
      this.zoomImage(1 - (.1/1.1))
    }
  }

  // Tell the Key object that this ImageViewer ("this") wants to listen
  // for Key events.
  Key.addListener(this);
  
  // Make a note that zooming is enabled.
  this.zoomEnabled = true;
}

/**
 * Method: ImageViewer.disableZoom()
 *   Desc: Turns off zooming.
 */
_global.ImageViewer.prototype.disableZoom = function () {
  // Make a note that zooming is disabled.
  this.zoomEnabled = false;

  // Tell the Key object that the image viewer (this) no longer
  // wants to listen for Key events.
  return Key.removeListener(this);
}



/**
 * Method: ImageViewer.centerImage()
 *   Desc: Centers the image in the image viewer.
 */
_global.ImageViewer.prototype.centerImage = function () {
  // A convenient reference to the image movie clip.
  var image = this.container.imageHolder_mc.image_mc;
  
  // Reposition the image in the center of the mask clip.
  image._x = (this.container.mask_mc._width / 2)
             - (image._width / 2);
  image._y = (this.container.mask_mc._height / 2)
             - (image._height / 2);
}

/**
 * Method: ImageViewer.zoomImage()
 *   Desc: Increases or decreases the size of the image.
 * Params: 
 *   zoomfactor  The amount to zoom, as a floating point number.
 *               For example, 2.0 doubles the image size, 
 *               while 0.5 halves the image size.
 */
_global.ImageViewer.prototype.zoomImage = function (zoomfactor) {
  // A convenient reference to the image movie clip.
  var image = this.container.imageHolder_mc.image_mc;

  // Calculate offset of image.
  centerX = this.container.mask_mc._width / 2;
  centerY = this.container.mask_mc._height / 2;
  var xOffset = centerX - image._x;
  var yOffset = centerY - image._y;

  // Scale the image.
  image._xscale = image._yscale *= zoomfactor;

  // Adjust image position so the previous center is still centered.
  image._x += xOffset - xOffset*zoomfactor;
  image._y += yOffset - yOffset*zoomfactor;

  // If the image is now smaller than the viewer (either horizontally
  // or vertically), center it.
  if ((this.container.mask_mc._width > image._width)
      || (this.container.mask_mc._height > image._height))
  {
    this.centerImage();
  } else {
    // If there's whitespace between the image and either horizontal border,
    // position the image flush with that border. This prevents the image
    // from zooming out of the viewer.
    if (this.container.mask_mc.getBounds(this.container).xMax >
      image.getBounds(this.container).xMax) {
      image._x = this.container.mask_mc.getBounds(this.container).xMax - image._width;
    } else if (this.container.mask_mc.getBounds(this.container).xMin <
      image.getBounds(this.container).xMin) {
      image._x = 0;
    }

    // If there's whitespace between the image and either vertical border,
    // position the image flush with that border. This prevents the image
    // from zooming out of the viewer.
    if (this.container.mask_mc.getBounds(this.container).yMax >
      image.getBounds(this.container).yMax) {
      image._y = this.container.mask_mc.getBounds(this.container).yMax - image._height;
    } else if (this.container.mask_mc.getBounds(this.container).yMin <
      image.getBounds(this.container).yMin) {
      image._y = 0;
    }
  }
}


/**
 * Method: ImageViewer.setScaleMode()
 *   Desc: Determines the scaling of a newly loaded image.
 * Params:
 *   scaleMode   The new scale mode. Must be one of: 
 *               "noScale" - don't scale the image
 *               "scaleImageToViewer" - scale the image to fit the viewer
 *               "scaleViewerToImage" - scale the viewer to fit the image
 */
_global.ImageViewer.prototype.setScaleMode = function (scaleMode) {
  // If an invalid scaleMode was passed in, quit. Return false
  // so the transgressor knows the error of their ways.
  if (scaleMode != "noScale"
      && scaleMode != "scaleImageToViewer"
      && scaleMode != "scaleViewerToImage") {
    return false;
  }

  // Set the scale mode.
  this.scaleMode = scaleMode;
  
  // Let the caller know the setting succeeded.
  return true;
}

/**
 * Method: ImageViewer.getScaleMode()
 *   Desc: Returns the scale mode setting. See setScaleMode().
 */
_global.ImageViewer.prototype.getScaleMode = function (scaleMode) {
  return this.scaleMode;
}




